using System;
using System.Collections.Generic;
using System.Configuration;
using System.Text;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
    public abstract class NamedObjectInjectionFactory<TObjectToCreate, TObjectConfiguration> : InjectionFactory<TObjectToCreate, string>, INamedObjectFactory<TObjectToCreate>
    {
        public override TObjectToCreate Create(string name, IConfigurationSource configurationSource)
        {
            if (string.IsNullOrEmpty(name))
            {
                throw new ArgumentException(string.Format(Resources.Culture, Resources.ExceptionConfigurationNameNullOrEmpty, this.GetType().Name));
            }

            TObjectConfiguration objectConfiguration;
            try
            {
                objectConfiguration = GetNamedConfiguration(name, configurationSource);
            }
            catch (Exception exception)
            {
                throw new InvalidOperationException(string.Format(Resources.Culture, Resources.ExceptionNamedConfigurationNotFound, name, this.GetType().Name), exception);
            }
            if (objectConfiguration == null)
            {
                throw new InvalidOperationException(string.Format(Resources.Culture, Resources.ExceptionNamedConfigurationNotFound, name, this.GetType().Name));
            }

            return Create(objectConfiguration, configurationSource);
        }

        protected virtual TObjectToCreate Create(TObjectConfiguration objectConfiguration, IConfigurationSource configurationSource)
        {
            IObjectFactory<TObjectToCreate, TObjectConfiguration> factory = new InjectionFactory<TObjectToCreate, TObjectConfiguration>();
            
            return factory.Create(objectConfiguration, configurationSource);
        }

        /// <summary>
        /// When overriden by a class, gets the named <typeparamref name="TObjectConfiguration"/> which is the configuraiton for the object to create from the configuration source.
        /// </summary>
        /// <param name="name">The name of the configured <typeparamref name="TObjectToCreate"/> to retrieve.</param>
        /// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
        /// <returns>tThe named <typeparamref name="TObjectConfiguration"/> which is the configuraiton for the object to create from the configuration source.</returns>
        protected abstract TObjectConfiguration GetNamedConfiguration(string name, IConfigurationSource configurationSource);

        /// <summary>
        /// Gets the name of the default configured instance to create from the configuration source.
        /// </summary>
        /// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
        /// <returns>The name of the default configured instance.</returns>
        /// <remarks>
        /// The default return is an empty string.
        /// </remarks>
        protected virtual string GetDefaultInstanceName(IConfigurationSource configurationSource)
        {
            return string.Empty;
        }

        #region INamedObjectFactory<TObjectToCreate,string> Members

        public TObjectToCreate CreateDefault()
        {
            return CreateDefault(new SystemConfigurationSource());
        }

        public virtual TObjectToCreate CreateDefault(IConfigurationSource configurationSource)
        {
            return Create(GetDefaultInstanceName(configurationSource), configurationSource);
        }

        #endregion
    }
}
